#region Includes
using GroundhogLearning.GameKit.Utils;
using UnityEngine;
using UnityEngine.Events;
#endregion

namespace GroundhogLearning.GameKit
{
    [AddComponentMenu(EditorMenuConfig.CATEGORY_TRANSFORMS + "Move To")]
    public class MoveTo : MonoBehaviour
    {
        #region Variables

        [Header("Assets")]
        [SerializeField, Tooltip("Target object to move to")]
        private Transform _targetObject;

        [Header("Configuration")]
        [SerializeField, Tooltip("If true, movement begins automatically when the scene starts.")]
        private bool _autoStart = true;

        [SerializeField, Tooltip("Movement speed in units per second.")]
        private float _speed = 1f;

        [SerializeField, Tooltip("Offset from the initial position that defines the target position.")]
        private Vector3 _targetOffset = new(0f, 0f, 0f);

        [SerializeField, Tooltip("If true, the object will move back and forth between start and target positions.")]
        private bool _loop = true;

        [SerializeField, Tooltip("Optional wait time at each end before reversing direction (if looping).")]
        private float _pauseTime = 0f;

        [SerializeField, Tooltip("If true, objects on top of this platform will move along with it.")]
        private bool _carryObjects = true;

        [Header("Events")]
        [SerializeField, Tooltip("Event raised when the object reaches the target position")]
        private UnityEvent OnTargetPositionReached;

        [SerializeField, Tooltip("Event raised when the object reaches the initial position")]
        private UnityEvent OnInitialPositionReached;


        public bool IsMoving => _isMoving;
        public void StartMoving() => _isMoving = true;
        public void StopMoving() => _isMoving = false;


        private Vector3 _initialPosition;
        private Vector3 _targetPosition;
        private Vector3 _currentTargetPosition;
        private Vector3 _lastPosition;
        private bool _isMoving;
        private float _pauseTimer;

        #endregion

        private void Start()
        {
            _initialPosition = transform.position;
            _targetPosition = _targetObject != null ? _targetObject.position : _initialPosition + _targetOffset;
            _currentTargetPosition = _targetPosition;
            _lastPosition = transform.position;

            if (_autoStart)
            {
                _isMoving = true;
            }
        }
        private void FixedUpdate()
        {
            if (!_isMoving) return;

            // Calculate delta movement
            _lastPosition = transform.position;

            transform.position = Vector3.MoveTowards(transform.position, _currentTargetPosition, _speed * Time.fixedDeltaTime);

            var movementDelta = transform.position - _lastPosition;

            if (_carryObjects && movementDelta.sqrMagnitude > 0f)
            {
                CarryObjects(movementDelta);
            }

            if (Vector3.Distance(transform.position, _currentTargetPosition) < 0.001f)
            {
                if (_loop)
                {
                    if (_pauseTime > 0f)
                    {
                        _pauseTimer += Time.deltaTime;
                        if (_pauseTimer < _pauseTime) return;
                        _pauseTimer = 0f;
                    }

                    if (_currentTargetPosition == _targetPosition)
                    {
                        _currentTargetPosition = _initialPosition;
                        OnTargetPositionReached?.Invoke();
                    }
                    else
                    {
                        _currentTargetPosition = _targetPosition;
                        OnInitialPositionReached?.Invoke();
                    }
                }
                else
                {
                    _isMoving = false;
                }
            }
        }


        /// <summary>
        /// Moves objects standing on top of this platform.
        /// </summary>
        private void CarryObjects(Vector3 delta)
        {
            // Cast upward a little bit from the platform to find objects on top
            var bounds = GetComponent<Collider>()?.bounds;
            if (bounds == null) return;

            var size = bounds.Value.size;
            var center = bounds.Value.center + Vector3.up * 0.05f;

            // Overlap small area above platform
            var hits = Physics.OverlapBox(center, size / 2f, transform.rotation);

            foreach (var hit in hits)
            {
                if (hit.attachedRigidbody && hit.attachedRigidbody.gameObject != gameObject)
                {
                    // Apply delta movement to rigidbody
                    hit.attachedRigidbody.position += delta;
                }
                else if (hit.TryGetComponent<CharacterController>(out var controller))
                {
                    // Move character controller manually
                    controller.Move(delta);
                }
            }
        }


#if UNITY_EDITOR
        private void OnDrawGizmosSelected()
        {
            Vector3 start = Application.isPlaying ? _initialPosition : transform.position;
            Vector3 end = _targetObject != null ? _targetObject.position : start + _targetOffset;

            Gizmos.color = Color.cyan;
            Gizmos.DrawLine(start, end);
            Gizmos.DrawSphere(end, 0.1f);
        }
#endif
    }
}
